--========================================================--
-- Unseasonal Weather - Radio Forecasting (Client relay)
-- Reads pending forecasts from ModData and emits via DynamicRadio.
--========================================================--

if not UW_RF then UW_RF = {} end

local UW_RF_MD_KEY = "UW_RF_PENDING"

------------------------------------------------------------
-- Player iteration helpers
------------------------------------------------------------
local function UW_RF_forEachPlayer(cb)
    local list = getOnlinePlayers and getOnlinePlayers()
    if list and list.size and list:size() > 0 then
        for i = 0, list:size() - 1 do
            cb(list:get(i))
        end
        return
    end

    if getPlayer then
        cb(getPlayer())
    end
end

------------------------------------------------------------
-- Walkie handling (client only)
------------------------------------------------------------
local function UW_RF_playerHasActiveRadio(player)
    if not player then return false end

    local inv = player:getInventory()
    if not inv then return false end

    local items = inv:getItems()
    if not items then return false end

    for i = 0, items:size() - 1 do
        local item = items:get(i)
        if item and item.getCategory and item:getCategory() == "Radio" then
            if item.isActivated and item:isActivated() then
                return true
            end
        end
    end

    return false
end

local function UW_RF_anyActiveRadioPlayer()
    local holder = nil
    UW_RF_forEachPlayer(function(player)
        if holder then return end
        if UW_RF_playerHasActiveRadio(player) then
            holder = player
        end
    end)
    return holder
end

local function UW_RF_debugEnabled()
    if not SandboxVars or not SandboxVars.UW_RadioForecasting then
        return false
    end
    return SandboxVars.UW_RadioForecasting.DebugLogging == true
end

local function UW_RF_walkieEnabled()
    if not SandboxVars or not SandboxVars.UW_RadioForecasting then
        return true
    end
    return SandboxVars.UW_RadioForecasting.WalkieHints ~= false
end

local function UW_RF_walkieMessage(key)
    if not key then return end
    if not UW_RF_walkieEnabled() then return end

    local msg = getText(key)
    if not msg or msg == "" then
        msg = key
    end

    UW_RF_forEachPlayer(function(player)
        if not player then return end

        local inv = player:getInventory()
        if not inv then return end

        local items = inv:getItems()
        if not items then return end

        for j = 0, items:size() - 1 do
            local it = items:get(j)
            if it and it.getCategory and it:getCategory() == "Radio" and it.isActivated and it:isActivated() then
                player:Say("[WX] " .. msg)
                break
            end
        end
    end)
end

------------------------------------------------------------
-- Broadcast helper (client)
------------------------------------------------------------
local function UW_RF_ClientTryBroadcast()
    local md = ModData.getOrCreate(UW_RF_MD_KEY)
    local forecast = md.pending
    if not forecast then return end

    local gt = getGameTime()
    local now = gt and gt:getWorldAgeHours() or nil
    if not forecast.key then
        md.pending = nil
        ModData.transmit(UW_RF_MD_KEY)
        return
    end

    if forecast.expires and now and now > forecast.expires then
        md.pending = nil
        ModData.transmit(UW_RF_MD_KEY)
        return
    end

    local targetHour = forecast.hour
    if targetHour == nil then
        md.pending = nil
        ModData.transmit(UW_RF_MD_KEY)
        return
    end

    local currentHour   = gt and gt:getHour() or 0
    local currentMinute = gt and gt:getMinutes() or 0
    local scheduledMinute = forecast.minute or 0

    if not forecast.jitter then
        forecast.jitter = ZombRand(1, 6) -- 1-5 minutos
        md.pending = forecast
        ModData.transmit(UW_RF_MD_KEY)
    end

    -- Si hay clima vanilla activo, no cancelar: reprogramar cerca del presente y extender expiracion.
    local cm = getWorld and getWorld():getClimateManager()
    if cm and cm.isWeatherPeriodActive and cm:isWeatherPeriodActive() then
        local newHour = currentHour or targetHour
        local newMinute = (currentMinute or scheduledMinute) + 1
        if newMinute >= 60 then
            newMinute = newMinute - 60
            newHour = ((newHour or 0) + 1) % 24
        end
        forecast.hour = newHour
        forecast.minute = newMinute
        if now then
            forecast.expires = now + 1.0 -- extiende 1h para no caducar
        end
        md.pending = forecast
        ModData.transmit(UW_RF_MD_KEY)
        if UW_RF_debugEnabled() then
            print(string.format("[UW_RF][CLIENT] Clima vanilla activo, difiriendo a %02d:%02d (exp=%.2f)", newHour or -1, newMinute or -1, forecast.expires or -1))
        end
        return
    end

    local jitter = forecast.jitter or 0
    local startMinuteAbs = (targetHour * 60) + scheduledMinute + jitter
    if startMinuteAbs < 0 then startMinuteAbs = 0 end

    local windowMinutes = 30
    local endMinuteAbs = startMinuteAbs + windowMinutes

    local currentAbs = (currentHour * 60) + currentMinute
    local wrap = 24 * 60

    startMinuteAbs = startMinuteAbs % wrap
    endMinuteAbs   = endMinuteAbs % wrap
    currentAbs     = currentAbs % wrap

    local withinWindow = false
    if startMinuteAbs <= endMinuteAbs then
        withinWindow = currentAbs >= startMinuteAbs and currentAbs <= endMinuteAbs
    else
        withinWindow = currentAbs >= startMinuteAbs or currentAbs <= endMinuteAbs
    end

    if not withinWindow then
        if UW_RF_debugEnabled() then
            print(string.format("[UW_RF][CLIENT] Skip window cur=%d:%02d start=%d end=%d jitter=%d", currentHour, currentMinute, math.floor(startMinuteAbs/60), endMinuteAbs % 60, jitter))
        end
        return
    end

    local listener = UW_RF_anyActiveRadioPlayer()

    -- Build lines using shared messages (key-driven only)
    local lines = nil
    if forecast.key then
        lines = UW_RF_GetLinesForKey and UW_RF_GetLinesForKey(forecast.key)
    end

    if lines then
        if UW_RF_debugEnabled() then
            local dbgText = getText(forecast.key) or ""
            print(string.format("[UW_RF][CLIENT] Emit broadcast key=%s lines=%d text=\"%s\"", tostring(forecast.key), #lines, dbgText))
        end
        UW_RF_EmitBroadcast(lines)
    else
        if UW_RF_debugEnabled() then
            print(string.format("[UW_RF][CLIENT] No lines built for key=%s", tostring(forecast.key)))
        end
    end

    if listener and forecast.walkie ~= false and forecast.walkieKey then
        UW_RF_walkieMessage(forecast.walkieKey)
    end

    if SandboxVars
        and SandboxVars.UW_RadioForecasting
        and SandboxVars.UW_RadioForecasting.VerbalizeForecast
        and listener
    then
        local sayText = forecast.key and getText(forecast.key) or nil
        if sayText and sayText ~= "" then
            listener:Say(sayText)
        end
    end

    if UW_RF_debugEnabled() then
        local dbgText = getText(forecast.key) or ""
        print(string.format("[UW_RF][CLIENT] Broadcast text=\"%s\"", dbgText))
    end

    md.pending = nil
    ModData.transmit(UW_RF_MD_KEY)

    if not UW_RF_debugEnabled() then
        print(string.format("[UW_RF][CLIENT] Forecast broadcasted (day %s @ %s) cat=%s", tostring(forecast.day), tostring(forecast.hour), tostring(forecast.cat)))
    end
end

------------------------------------------------------------
-- Events (client)
------------------------------------------------------------
Events.OnLoadRadioScripts.Add(UW_RF_ClientTryBroadcast)
Events.EveryOneMinute.Add(UW_RF_ClientTryBroadcast)

print("[UW_RF] Radio Forecasting Client relay loaded.")
